program JACOBI_EIGEN;
{--------------------------------------------------------------------}
{  Alg11'3.pas   Pascal program for implementing Algorithm 11.3      }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 11.3 (Jacobi Iteration for Eigenvalues & Eigenvectors). }
{  Section   11.3, Jacobi's Method, Page 571                         }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxR = 10;
    MaxS = 50;
  type
    SubS = 1..MaxR;
    VECTOR = array[SubS] of real;
    MATRIX = array[SubS, SubS] of real;
    LETTER = string[4];
    LETTERS = string[200];
    Status = (Done, Working);
    DoSome = (Go, New, Stop);
    MatType = (LowerT, Square, UpperT);
    Method = (Cyclic, Original);
    Process = (Auto, Manual, Observe);

  var
    CountR, CountS, I, InRC, Inum, N, P, Q, Sub: integer;
    XP, XQ, YP, YQ: VECTOR;
    A, A1, V: MATRIX;
    Apq, C, Epsilon, MaxA, S, Rnum, T: real;
    Ach, Ans: LETTER;
    Mess: LETTERS;
    Stat: Status;
    DoMo: DoSome;
    Mtype: MatType;
    Meth: Method;
    Proc: Process;

  procedure Routput (C, S: real);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('The rotation matrix for zeroing out  A(', P : 1, ',', Q : 1, ') =', Apq, '  is');
    WRITELN;
    WRITELN('            (               )     (                              )');
    WRITELN('            (   c       s   )     (  ', C : 10 : 7, '      ', S : 10 : 7, '  )');
    WRITELN('      R  =  (               )  =  (                              )');
    WRITELN('            (  -s       c   )     (  ', -S : 10 : 7, '      ', C : 10 : 7, '  )');
    WRITELN('            (               )     (                              )');
  end;                                       {End of procedure Routput}

  procedure Aoutput (Ach: LETTER; A: MATRIX; N: integer);
    var
      Digits, Mdigits, C, R: integer;
      Log10: real;
  begin
    Log10 := LN(10);
    WRITELN;
    WRITELN('The matrix  ', Ach, '  is:');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to N - 1 do
          begin
            Digits := 7;
            if A[R, C] <> 0 then
              Mdigits := 1 + TRUNC(LN(ABS(A[R, C])) / Log10);
            if A[R, C] < 0 then
              Mdigits := Mdigits + 1;
            if Mdigits < 7 then
              Mdigits := 7;
            Digits := 14 - Mdigits;
            WRITE(A[R, C] : 15 : Digits, ' ');
          end;
        Digits := 7;
        if A[R, N] <> 0 then
          Mdigits := 1 + TRUNC(LN(ABS(A[R, N])) / Log10);
        if A[R, N] < 0 then
          Mdigits := Mdigits + 1;
        if Mdigits < 7 then
          Mdigits := 7;
        Digits := 14 - Mdigits;
        WRITE(A[R, N] : 15 : Digits);
        if N > 5 then
          WRITELN;
      end;
    WRITELN;
  end;                                       {End of procedure Aoutput}

  procedure ZEROPQ (var A, V: MATRIX; var C, S: real; N: integer; P, Q: integer);
    var
      I, J, Pm, Qm: integer;
      MaxA, T, Theta: real;
      XP, XQ, YP, YQ: VECTOR;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Theta := (A[Q, Q] - A[P, P]) / (2 * A[P, Q]);
    if Theta < 1E19 then
      T := 1 / (ABS(Theta) + SQRT(Theta * Theta + 1))
    else
      T := 1 / ABS(2 * Theta);
    if Theta < 0 then
      T := -T;
    C := 1 / SQRT(T * T + 1);
    S := C * T;
    for I := 1 to N do
      begin
        XP[I] := A[I, P];
        XQ[I] := A[I, Q];
      end;
    A[P, Q] := 0;
    A[Q, P] := 0;
    A[P, P] := C * C * XP[P] + S * S * XQ[Q] - 2 * C * S * XQ[P];
    A[Q, Q] := S * S * XP[P] + C * C * XQ[Q] + 2 * C * S * XQ[P];
    for I := 1 to N do
      if (I <> P) and (I <> Q) then
        begin
          A[I, P] := C * XP[I] - S * XQ[I];
          A[I, Q] := C * XQ[I] + S * XP[I];
          A[P, I] := A[I, P];
          A[Q, I] := A[I, Q];
        end;
    for I := 1 to N do
      begin
        YP[I] := V[I, P];
        YQ[I] := V[I, Q];
      end;
    for I := 1 to N do
      begin
        V[I, P] := C * YP[I] - S * YQ[I];
        V[I, Q] := S * YP[I] + C * YQ[I];
      end;
  end;

  procedure MAXPQ (A: MATRIX; N: integer; var P, Q: integer; var MaxA: real);
    var
      I, J, Pm, Qm: integer;
  begin
    MaxA := ABS(A[1, N]);
    Pm := 1;
    Qm := N;
    for I := 1 to N do
      for J := I + 1 to N do
        if ABS(A[I, J]) > MaxA then
          begin
            Pm := I;
            Qm := J;
            MaxA := ABS(A[I, J]);
          end;
    P := Pm;
    Q := Qm;
  end;

  procedure INITIALV (var V: MATRIX; N: integer);
    var
      I, J: integer;
  begin
    for I := 1 to N do
      for J := 1 to N do
        if I <> J then
          V[I, J] := 0
        else
          V[I, J] := 1;
  end;

  function RMS (A: MATRIX; N: integer): real;
    var
      J: integer;
      Sum: real;
  begin
    Sum := 0;
    for J := 1 to N do
      Sum := Sum + A[J, J] * A[J, J];
    RMS := SQRT(Sum / N);
  end;

  procedure SELECTPQ (A: MATRIX; N: integer; var P, Q: integer; var MaxA: real);
    type
      STATUS = (Enter, Done);
    var
      C, I, J, Pm, Qm, Ptemp, Qtemp: integer;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) do
      begin
        MAXPQ(A, N, P, Q, MaxA);
        Apq := A[P, Q];
        if Proc <> Manual then
          Stat := Done;
        if Proc = Manual then
          begin
            CLRSCR;
            Aoutput(Ach, A, N);
            WRITELN;
            WRITELN('     Choose an off diagonal element to "zero-out."');
            WRITELN;
            case N of
              2: 
                begin
                  WRITELN('     Select the row  P = 1,2');
                  WRITELN('        and  column  Q = 1,2');
                end;
              3:
                begin
                  WRITELN('     Select the row  P = 1,2,3');
                  WRITELN('        and  column  Q = 1,2,3');
                end;
              else
                begin
                  WRITELN('     Select the row  P = 1,2,...,', N : 2);
                  WRITELN('        and  column  Q = 1,2,...,', N : 2);
                end;
            end;
            WRITELN;
            WRITE('     ENTER  the  row P = ');
            READLN(Ptemp);
            if (1 <= Ptemp) and (Ptemp <= N) then
              P := PtemP;
            WRITE('     ENTER  column   Q = ');
            READLN(Qtemp);
            if (1 <= Qtemp) and (Qtemp <= N) then
              Q := QtemP;
            Apq := A[P, Q];
            WRITELN;
            WRITELN('     The element selected was  A(', P, ',', Q, ')  =', A[P, Q]);
            WRITELN;
            WRITE('     Is this the correct choice  <Y/N> ?  ');
            READLN(Resp);
            if (Resp <> 'N') and (Resp <> 'n') then
              Stat := Done;
            WRITELN;
          end;
      end;
  end;

  procedure ORIGINAL_JACOBI;
  begin
    CountR := 0;
    while Stat = Working do
      begin
        SELECTPQ(A, N, P, Q, MaxA);
        ZEROPQ(A, V, C, S, N, P, Q);
        CountR := CountR + 1;
        if Proc = Auto then
          WRITELN('               I am computing rotation  # ', CountR);
        if (Proc = Manual) or (Proc = Observe) then
          begin
            CLRSCR;
            Routput(C, S);
            Aoutput(Ach, A, N);
            if Proc = Observe then
              ;{DELAY(2000)}
          end;
        MAXPQ(A, N, P, Q, MaxA);
        if MaxA < Epsilon * RMS(A, N) then
          Stat := Done;
        if Proc = Manual then
          begin
            WRITELN;
            WRITE('     Want to zero out another element ? <Y/N>  ');
            READLN(Ans);
            if (Ans = 'N') or (Ans = 'n') then
              Stat := Done;
          end;
      end;
  end;

  procedure CYCLIC_JACOBI;
    var
      P, Q: integer;
  begin
    CountR := 0;
    CountS := 0;
    while (Stat <> Done) and (CountS <= MaxS) do
      begin
        CountS := CountS + 1;
        T := RMS(A, N);
        Stat := Done;
        for P := 1 to N - 1 do
          for Q := P + 1 to N do
            if (ABS(A[P, Q]) / T) > Epsilon then
              begin
                ZEROPQ(A, V, C, S, N, P, Q);
                CountR := CountR + 1;
                if Proc = Auto then
                  begin
                    WRITELN('     I am computing sweep  # ', CountS, '  ,  rotation  # ', CountR);
                  end;
                if (Proc = Observe) then
                  begin
                    CLRSCR;
                    Routput(C, S);
                    Aoutput(Ach, A, N);
                    ;{DELAY(2000)}
                  end;
                Stat := Working;
              end;
      end;
  end;

  procedure INPUTMATRIX (var Ach: LETTER; var A, A1: MATRIX; N, InRC: integer);
    var
      Count, C, CL, CU, K, R, RL, RU: integer;
      Z: VECTOR;
  begin
    for R := 1 to N do
      begin
        for C := 1 to N do
          begin
            A[R, C] := 0;
            A1[R, C] := A[R, C];
          end;
      end;
    WRITELN('     Input the elements of the ', N : 1, ' by ', N : 1, ' coefficient matrix  ', Ach);
    RL := 1;
    RU := N;
    CL := 1;
    CU := N;
    if (Mtype = LowerT) and (InRC = 1) then
      begin
        for R := 1 to N do
          begin
            WRITELN;
            case R of
              1:
                WRITELN('ENTER   A[1,1]');
              2:
                WRITELN('ENTER   A[2,1]   A[2,2]   on one row');
              3:
                WRITELN('ENTER   A[3,1]   A[3,2]   A[3,3]   on one row');
              else
                WRITELN('ENTER   A[', R : 1, ',1]   A[', R : 1, ',2]  ...  A[', R : 1, ',', R : 1, ']   on one row');
            end;
            WRITELN;
            for K := 1 to R do
              Z[K] := 0;
            case R of
              1: 
                READLN(Z[1]);
              2: 
                READLN(Z[1], Z[2]);
              3: 
                READLN(Z[1], Z[2], Z[3]);
              4: 
                READLN(Z[1], Z[2], Z[3], Z[4]);
              5: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5]);
              6: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6]);
              7: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7]);
              8: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8]);
              9: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9]);
              10: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9], Z[10]);
            end;
            for C := 1 to R do
              begin
                A[R, C] := Z[C];
                A1[R, C] := A[R, C];
              end;
          end;
      end;
    if (Mtype = UpperT) and (InRC = 1) then
      begin
        for R := 1 to N do
          begin
            WRITELN;
            case R of
              1: 
                WRITELN('ENTER   A[1,1]   A[1,2]  ...  A[1,', N : 1, ']   on one row');
              2: 
                WRITELN('ENTER   A[2,2]   A[2,3]  ...  A[2,', N : 1, ']   on one row');
              else
                WRITELN('ENTER   A[', R : 1, ',', R : 1, ']   A[',R:1,',',R+1:1,']  ...  A[',R:1,',',N:1,']   on one row');
            end;
            WRITELN;
            for K := 1 to N do
              Z[K] := 0;
            case N - R + 1 of
              1: 
                READLN(Z[1]);
              2: 
                READLN(Z[1], Z[2]);
              3: 
                READLN(Z[1], Z[2], Z[3]);
              4: 
                READLN(Z[1], Z[2], Z[3], Z[4]);
              5: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5]);
              6: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6]);
              7: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7]);
              8: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8]);
              9: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9]);
              10: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9], Z[10]);
            end;
            for C := R to N do
              begin
                A[R, C] := Z[C - R + 1];
                A1[R, C] := A[R, C];
              end;
          end;
      end;
    if (InRC = 1) and (Mtype <> LowerT) and (Mtype <> UpperT) then
      begin
        for R := 1 to N do
          begin
            WRITELN;
            WRITELN('ENTER all the coefficients of row ', R, ' on one row');
            WRITELN;
            for K := 1 to N do
              Z[K] := 0;
            case N of
              1: 
                READLN(Z[1]);
              2: 
                READLN(Z[1], Z[2]);
              3: 
                READLN(Z[1], Z[2], Z[3]);
              4: 
                READLN(Z[1], Z[2], Z[3], Z[4]);
              5: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5]);
              6: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6]);
              7: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7]);
              8: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8]);
              9: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9]);
              10: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9], Z[10]);
            end;
            for C := 1 to N do
              begin
                A[R, C] := Z[C];
                A1[R, C] := A[R, C];
              end;
          end;
      end;
    if (InRC = 2) then
      begin
        for R := 1 to N do
          begin
            WRITELN;
            WRITELN('     ENTER the coefficients of row ', R);
            WRITELN;
            if Mtype = LowerT then
              CU := R;
            if Mtype = UpperT then
              CL := R;
            for C := CL to CU do
              begin
                WRITE('     A(', R, ',', C, ') = ');
                READLN(A[R, C]);
                A1[R, C] := A[R, C];
              end;
          end;
      end;
    if (InRC = 3) then
      begin
        for C := 1 to N do
          begin
            WRITELN;
            WRITELN('     ENTER the coefficients of column ', C);
            WRITELN;
            if Mtype = LowerT then
              RL := C;
            if Mtype = UpperT then
              RU := C;
            for R := RL to RU do
              begin
                WRITE('     A(', R, ',', C, ') = ');
                READLN(A[R, C]);
                A1[R, C] := A[R, C];
              end;
          end;
      end;
    if Mtype = LowerT then
      begin
        for R := 1 to N do
          for C := 1 to R do
            begin
              A[C, R] := A[R, C];
              A1[C, R] := A[C, R];
            end;
      end;
    if Mtype = UpperT then
      begin
        for R := 1 to N do
          for C := R to N do
            begin
              A[C, R] := A[R, C];
              A1[C, R] := A[C, R];
            end;
      end;
    Mtype := Square;
  end;                                   {End of procedure INPUTMATRIX}

  procedure REFRESH (var A: MATRIX; A1: MATRIX; N: integer);
    var
      C, R: integer;
  begin
    for R := 1 to N do
      begin
        for C := 1 to N do
          begin
            A[R, C] := A1[R, C];
          end;
      end;
  end;

  procedure PrintResults (A, A1, V: MATRIX; N: integer; Epsilon, MaxA: real);
    var
      C, R: integer;
      A0: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('The matrix  A  is:');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to N - 1 do
          WRITE(A1[R, C] : 15 : 8, ' ');
        WRITE(A1[R, N] : 15 : 8);
        if N > 5 then
          WRITELN;
      end;
    WRITELN;
    WRITELN;
    WRITELN('The eigenvalues of  A  are:');
    for R := 1 to N - 1 do
      WRITE('     X[', R : 1, ']       ');
    WRITE('     X[', N : 1, ']');
    WRITELN;
    for R := 1 to N - 1 do
      WRITE(A[R, R] : 15 : 8, ' ');
    WRITE(A[N, N] : 15 : 8);
    if N > 5 then
      WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('The matrix of eigenvectors  V  is:');
    for R := 1 to N - 1 do
      WRITE('     V[', R : 1, ']       ');
    WRITE('     V[', N : 1, ']');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to N - 1 do
          WRITE(V[R, C] : 15 : 8, ' ');
        WRITE(V[R, N] : 15 : 8);
        if N > 5 then
          WRITELN;
      end;
    WRITELN;
    if (MaxA > Epsilon * RMS(A, N)) and (Meth = Original) then
      begin
        WRITELN;
        WRITELN('The iteration did NOT converge!');
      end;
    if (MaxS < CountS) and (Meth = Cyclic) then
      begin
        WRITELN;
        WRITELN('The iteration did NOT converge!');
      end;
  end;                                {End of procedure PrintResults}

  procedure CHANGEMATRIX (Ach: LETTER; var A, A1: MATRIX; N: integer);
    type
      STATUS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      Count, C, I, J, K, R: integer;
      Valu: real;
      Resp: LETTER;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) or (Stat = Bad) do
      begin
        CLRSCR;
        Aoutput(Ach, A1, N);
        WRITELN;
        Stat := Enter;
        for I := 1 to N do
          for J := I + 1 to N do
            begin
              if A[I, J] <> A[J, I] then
                begin
                  R := I;
                  C := J;
                  Stat := Bad;
                end;
            end;
        if (Stat = Bad) then
          begin
            WRITELN('The matrix is NOT symmetric, you must change an element.');
          end;
        if (Stat <> Bad) then
          begin
            WRITE('Do you want to make a change in the matrix ? <Y/N> ');
            READLN(Resp);
          end;
        if (Resp = 'Y') or (Resp = 'y') or (Stat = Bad) then
          begin
            WRITELN;
            WRITELN('     To change a coefficient select');
            case N of
              2: 
                begin
                  WRITELN('        the row    R = 1,2');
                  WRITELN('        and column C = 1,2');
                end;
              3: 
                begin
                  WRITELN('        the row    R = 1,2,3');
                  WRITELN('        and column C = 1,2,3');
                end;
              else
                begin
                  WRITELN('        the row    R = 1,2,...,', N : 2);
                  WRITELN('        and column C = 1,2,...,', N : 2);
                end;
            end;
            WRITELN;
            WRITE('     ENTER the row R = ');
            READLN(R);
            WRITE('     ENTER column  C = ');
            READLN(C);
            if (1 <= R) and (R <= N) and (1 <= C) and (C <= N) then
              begin
                WRITELN;
                WRITELN('     The current value is   A(', R, ',', C, ')  =', A[R, C]);
                if A[R, C] <> A[C, R] then
                  begin
                    WRITELN('     Which is NOT equal to  A(', C, ',', R, ')  =', A[C, R]);
                    WRITELN('     The computer will set  A(', R, ',', C, ')  =  A(', C, ',', R, ')');
                    WRITELN;
                  end;
                WRITE('     ENTER the  NEW  value  A(', R, ',', C, ') = ');
                READLN(A[R, C]);
                A1[R, C] := A[R, C];
                A[C, R] := A[R, C];
                A1[C, R] := A[C, R];
              end;
          end
        else
          Stat := Done;
      end;
  end;

  procedure CHOOSEMETH;
    var
      I: integer;
  begin
    CLRSCR;
    WRITELN('     Choose a strategy for selecting the off diagonal element to');
    WRITELN;
    WRITELN('annihilate in the construction process.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('<1>  Jacobi`s original strategy.');
    WRITELN;
    WRITELN('     Select  p,q  so that  |a  |  =  max |a  |. ');
    WRITELN('                             pq      i<j   ij   ');
    WRITELN;
    WRITELN;
    WRITELN('<2>  The cyclic Jacobi method.');
    WRITELN;
    WRITELN('     Sweep through the matrix and annihilate elements in the strict');
    WRITELN;
    WRITELN('     order  a  , a  ,..., a  ; a  , a  ,..., a  ;...; a     .');
    WRITELN('             12   13       1N   23   24       2N       N-1,N              ');
    WRITELN;
    WRITELN;
    WRITE('     SELECT the strategy  < 1 or 2 > ?  ');
    I := 1;
    READLN(I);
    if (I < 1) or (2 < I) then
      I := 1;
    if I = 1 then
      Meth := Original;
    if I = 2 then
      Meth := Cyclic;
  end;

  procedure MESSAGE (var InRC: integer; var Mtype: MatType);
    var
      I: integer;
      Ans: LETTER;
  begin
    CLRSCR;
    WRITELN('                   JACOBI`S  METHOD FOR EIGENVECTORS');
    WRITELN;
    WRITELN;
    WRITELN('     Assume that A is an N by N real symmetric matrix.  Then A has a full');
    WRITELN;
    WRITELN;
    WRITELN('set of eigenvectors V , V ,..., V .  Jacobi`s method of iteration is used');
    WRITELN('                     1   2       N ');
    WRITELN;
    WRITELN('to find the eigenvalues and eigenvectors of A.  Let A = A , and construct');
    WRITELN('                                                     1 ');
    WRITELN('                                                             T ');
    WRITELN('a sequence of orthogonal matrices  { R  }  such that  D  =  R  A  R .  The');
    WRITELN('                                      j                j     j  j  j ');
    WRITELN;
    WRITELN('sequence  { D  }  converges to the diagonal matrix  D  of eigenvalues, and');
    WRITELN('             j ');
    WRITELN;
    WRITELN('the sequence { V  = R R ...R  }  converges to the matrix of eigenvectors.');
    WRITELN('                j    1 2    j  ');
    WRITELN;
    WRITELN;
    WRITE('                     Press the  <ENTER>  key.  ');
    READLN(Ans);
    CHOOSEMETH;
    CLRSCR;
    for I := 1 to 5 do
      WRITELN;
    WRITELN('     Now you must choose how the symmetric matrix A will be input.');
    WRITELN;
    WRITELN('You can enter all the elements or only the lower or upper portion.?');
    WRITELN;
    WRITELN('If you enter a portion of the matrix then the other elements will');
    WRITELN;
    WRITELN('be computed by symmetry.');
    WRITELN;
    WRITELN;
    WRITELN('     < 1 > Enter the complete  N by N  symmetric matrix.');
    WRITELN;
    WRITELN;
    WRITELN('     < 2 > Enter the lower-triangular portion of the matrix.');
    WRITELN;
    WRITELN;
    WRITELN('     < 3 > Enter the upper-triangular portion of the matrix.');
    WRITELN;
    WRITELN;
    WRITE('           SELECT your choice for input  < 1 - 3 > ? ');
    I := 1;
    READLN(I);
    if (I < 1) or (3 < I) then
      I := 1;
    if I = 1 then
      Mtype := Square;
    if I = 2 then
      Mtype := LowerT;
    if I = 3 then
      Mtype := UpperT;
    CLRSCR;
    WRITELN;
    WRITELN('        Choose how you want to input the elements of the matrix.');
    WRITELN;
    WRITELN('    <1> Enter the elements of each row on one line separated by spaces, i.e.');
    WRITELN;
    WRITELN('        A(J,1)  A(J,2)  ...  A(J,N)           for J=1,2,...,N');
    WRITELN;
    WRITELN('    <2> Enter each element of a row on a separate line, i.e.');
    WRITELN;
    WRITELN('        A(J,1)');
    WRITELN('        A(J,2)');
    WRITELN('           .');
    WRITELN('           :');
    WRITELN('        A(J,N)     for J=1,2,...,N');
    WRITELN;
    WRITELN('    <3> Enter each element of a column on a separate line, i.e.');
    WRITELN;
    WRITELN('        A(1,K)');
    WRITELN('        A(2,K)');
    WRITELN('           .');
    WRITELN('           :');
    WRITELN('        A(N,K)     for K=1,2,...,N');
    WRITELN;
    WRITE('        SELECT <1 - 3> ? ');
    InRC := 3;
    READLN(InRC);
    if (InRC <> 1) and (InRC <> 2) and (InRC <> 3) then
      InRC := 2;
  end;                                  {End of procedure MESSAGE}

  procedure INPUTS (var A, A1: MATRIX; var N, InRC: integer);
    var
      C, I, R: integer;
  begin
    CLRSCR;
    WRITELN('    We will now proceed with Jacobi`s method of iteration to find the');
    WRITELN;
    WRITELN('set of  N  eigenvectors  V , V ,..., V   for the symmetric matrix  A.');
    WRITELN('                          1   2       N ');
    WRITELN;
    WRITELN('           A  must be a symmetric matrix of dimension  N by N.');
    WRITELN;
    WRITELN('          {N  must be an integer between 1 and 10}');
    WRITELN;
    WRITE('    ENTER  N  = ');
    N := 2;
    READLN(N);
    if (N < 1) then
      N := 1;
    if (N > 10) then
      N := 10;
    WRITELN;
    WRITELN;
    WRITELN('     For each sweep throughout the matrix, the computed value  T');
    WRITELN;
    WRITELN('is the [R.M.S.] average of the diagonal elements of the matrix  A.');
    WRITELN;
    WRITELN('Give the error criterion for annihilating off diagonal elements');
    WRITELN;
    WRITELN('     |d   |  >  T*Epsilon    for all q > p.');
    WRITELN('       p,q');
    WRITELN;
    WRITE('           ENTER  Epsilon  = ');
    Epsilon := 0.00000001;
    READLN(Epsilon);
    if Epsilon < 0.000000001 then
      Epsilon := 0.000000001;
    CLRSCR;
    Ach := 'A';
    INPUTMATRIX(Ach, A, A1, N, InRC);
  end;                                   {End of procedure INPUTS}

  procedure PROCESSES;
    var
      I: integer;
  begin
    CLRSCR;
    for I := 1 to 5 do
      WRITELN;
    WRITELN('          To what extent do you want to control the program?');
    WRITELN;
    WRITELN;
    WRITELN('          < 1 > The computer does it all automatically.');
    WRITELN;
    WRITELN;
    WRITELN('          < 2 > Computer selects, but we observe each step.');
    WRITELN;
    WRITELN;
    WRITELN('          < 3 > The user selects elements to zero-out manually.');
    WRITELN;
    WRITELN;
    WRITE('          SELECT your choice for input  < 1 - 3 > ? ');
    I := 1;
    READLN(I);
    if (I < 1) or (3 < I) then
      I := 3;
    if I = 1 then
      Proc := Auto;
    if I = 2 then
      Proc := Observe;
    if I = 3 then
      Proc := Manual;
  end;

  procedure DOMORE (var Stat: Status);
    var
      Resp: string[40];
  begin
    WRITELN;
    WRITE('Press the <ENTER> key.');
    READLN(Ans);
    WRITELN;
    WRITE('Want to solve A*X = B with a new vector  B ? <Y/N> ');
    READLN(Resp);
    Resp := COPY(Resp, 1, 1);
    if (Resp <> 'y') and (Resp <> 'Y') then
      Stat := Done;
  end;                                        {End of procedure DOMORE}

begin                                            {Begin Main Program}
  MESSAGE(InRC, Mtype);
  DoMo := Go;
  while (DoMo = Go) or (DoMo = New) do
    begin
      if DoMo = Go then
        INPUTS(A, A1, N, InRC)
      else
        begin
          WRITELN;
          WRITE('Want a completely new matrix ? <Y/N> ');
          READLN(Ans);
          if (Ans = 'Y') or (Ans = 'y') then
            INPUTS(A, A1, N, InRC)
          else
            REFRESH(A, A1, N);
          WRITELN;
        end;
      CHANGEMATRIX(Ach, A, A1, N);
      INITIALV(V, N);
      PROCESSES;
      Stat := Working;
      MAXPQ(A, N, P, Q, MaxA);
      CLRSCR;
      if Meth = Original then
        ORIGINAL_JACOBI;
      if Meth = Cyclic then
        CYCLIC_JACOBI;
      WRITELN;
      PrintResults(A, A1, V, N, Epsilon, MaxA);
      WRITELN;
      WRITE('Want  to solve  a new system ? <Y/N> ');
      READLN(Ans);
      if (Ans = 'Y') or (Ans = 'y') then
        begin
          DoMo := New;
          WRITELN;
          WRITE('Want  to  change  the method ? <Y/N> ');
          READLN(Ans);
          if (Ans = 'Y') or (Ans = 'y') then
            CHOOSEMETH;
        end
      else
        DoMo := Stop;
    end;
end.                                            {End of Main Program}

